/*
  BombingRun game for Unreal Tournament 3. 
  Two teams seek possesion of a single ball. Scoring is done by taking the
  ball to the enemy goal.
    
  Author: Mark Caldwell Aka W@rHe@d of The Reliquary.
  
  Epic did away with bombing run for UT3 so we brought it back.
  Due to portability issues, game was rewritten from scratch.
  Started with the UT3 VCTF game classes and transformed them into BR.
  Some very small portions of code from UT2004 were able to be ported in.
*/

class UTBRGame extends UTTeamGame;

const BAD_ROUTE_DIST = 999999999;

var Array<UTBRObjectiveInfo> ObjectiveInfo;

var string Version;
var UTBRBall TheBall;
var UTBRGoal Goals[2];
var int roundTimer;
var UTBRBallBase BallHomeBase;
var bool SpawnProtecting;
var class<UTLocalMessage> AnnouncerMessageClass;
var array<UTBRPlayerManager> PlayerManagerArray;
var UTBRPlayerManager LocalPlayerManager;
var bool IsInstagib;  //used to tell bot to try shooting ball more, as you die a lot in instagib so shooting ball needs to be done more.
var int savCountDown;
var bool Test_NoShoot, Test_NoJump, Test_NoHackPath, Test_ShowPath, Test_NoGoalJumpPad, 
         Test_GiveBall, TestInited, Test_NoScore, Test_NoWeaponVolumes,
         Test_NoBallReset, Test_ShowPathRatings, Test_NoBallPickup, Test_HealthyBots, Test_DestroyVehicles, 
         Test_NavigateWithBall, Test_BotVehicle, Test_ClanTags, Test_AllowExit,
         Test_JumpyBots, Test_NoNodes, Test_BotsLoaded, Test_NoLoopChecks, Test_NoSetupSpecialPathAbilities,
         Test_AlwaysRepath;
                 
var UTVehicle TestBotVehicle;

var UTBRSettings Settings;

var float SteerOnDuration, SteerOffDuration;
var Vector TestBotFocus;
var Actor BlockingActor;
var vector BlockingActorHitLocation;
var Actor MoveTarget;
var config RepNotify string SkillDescription;
var bool takesLead, increasesLead;
var bool useBRGravity;
var bool useBRGameSpeed;
var bool AdminRightsNotNeeded;


//Settings vars replicated to client
var bool GameEnded;
var bool InitedManagers;
var RepNotify bool DebugMatchStartup;
var bool IsPendingMatch;
var bool DebugHud;
//


struct TeamStartsStruct
{
    var Array<PlayerStart> StartPoints;
    var int ix;
};


simulated function PostBeginPlay()
{
    local int i;
    local Mutator M;
    local UTBRPlayerManager playerManager;  
    
    LogInternal(GetVersionMessage());

    //a hack to try to make UTVoice.uc play orb message instead of flag message.
    //no support for just passing the orb message we want.
    //UTVoice plays orb message if game class is onslaught.
    WorldInfo.GRI.GameClass = class'UTOnslaughtGame';
                
    Super(UTTeamGame).PostBeginPlay();
                  
    if( Role < ROLE_Authority )
    {
        WorldInfo.Game = self;
        
        ForEach WorldInfo.AllActors(class'UTBRPlayerManager', playerManager)
        {
            PlayerManagerArray[PlayerManagerArray.Length] = playerManager;
        }  
         
        return;
    }
       
    useBRGravity = (WorldInfo.WorldGravityZ == WorldInfo.DefaultGravityZ);
    useBRGameSpeed = WorldInfo.Game.GameSpeed == class'GameInfo'.default.GameSpeed;
                   
    for ( i=0; i<2; i++ )
    {
        Teams[i].HomeBase = Goals[i];
        UTBRTeamAI(Teams[i].AI).FriendlyGoal = Goals[i];
        UTBRTeamAI(Teams[i].AI).EnemyGoal = Goals[1-i];           
    }
            
    RegisterBall(TheBall);

    TheBall.SendHome(none); 
        
    SetTimer(1.0, True, 'BRGameTimer');
    
    for (M = BaseMutator; M != None; M = M.NextMutator)
    {
        IsInstagib = InStr(CAPS(M.name), "INSTAGIB") >= 0;
    }
    
    SetupObjectiveInfo();
    FindCloseActors();    
    
    /*
      0.1/0.2 is fairly smooth and accurate.
      01./0.3 is not as accurate and bot is more likely to drift over
      a goal and hit top rim.
    */
    SteerOnDuration = 0.10;
    SteerOffDuration = 0.20;

    ApplySettings();    
}

function bool NodeActivated(UTOnslaughtNodeObjective node)
{
    return (! node.IsNeutral()) && (node.Health >= (node.DamageCapacity / 2)); 
}

/** RatePlayerStart()
* Return a score representing how desireable a playerstart is.
* @param P is the playerstart being rated
* @param Team is the team of the player choosing the playerstart
* @param Player is the controller choosing the playerstart
* @returns playerstart score
*/
function float RatePlayerStart(PlayerStart P, byte Team, Controller Player)
{
  local UTOnslaughtNodeObjective node;
	
	if (UTWarfarePlayerStart(P) != none)
	{
      node = FindParentNode(P);
    	  
      if (node != none)
      {
         if ((! NodeActivated(node)) || (Node.DefenderTeamIndex != Team))
         {
             return -9;
         }      
      }
  }
      
	if ((UTTeamPlayerStart(P) != none) && (Team != UTTeamPlayerStart(P).TeamNumber))
	{
      return -9;
	}
	  
	return Super(UTGame).RatePlayerStart(P,Team,Player);
}

function UTOnslaughtNodeObjective ClosestNodeObjectiveTo(Actor A)
{
	local float Distance, BestDistance;
	local UTOnslaughtNodeObjective Best;
	local UTOnslaughtNodeObjective Node;	
	local int i;

  for (i = 0; i < ObjectiveInfo.Length; i++)
  {   
      node = UTOnslaughtNodeObjective(ObjectiveInfo[i].Objective);
      
      if (node != none)
      {
      	  if ((PlayerStart(A) != none) && (! node.bAssociatePlayerStarts))
         	{
              continue;
          }
          
    			Distance = VSize(A.Location - node.Location);
    			
     			if ( (Best == None) || (Distance < BestDistance) )
     			{
      				BestDistance = Distance;
      				Best = node;
     			}
	    }
	}
	
	return Best;
}

//find node that vehicle or playerstart A belongs to, if any
function UTOnslaughtNodeObjective FindParentNode(Actor A)
{
	local UTOnslaughtNodeObjective node;
	local int i, j;

  for (i = 0; i < ObjectiveInfo.Length; i++)
  {   
        node = UTOnslaughtNodeObjective(ObjectiveInfo[i].Objective);
        
        if (node != none)
        {
            if (UTVehicleFactory(A) != none)
            {
              for (j = 0; j < node.VehicleFactories.Length; j++)
              {
                  if (node.VehicleFactories[j] == A)
                  {
                      return node;
                  }
              }
            }

            if (PlayerStart(A) != none)
            {
              for (j = 0; j < node.PlayerStarts.Length; j++)
              {
                  if (node.PlayerStarts[j] == A)
                  {
                      return node;
                  }
              }            
            }
        }
  }
  
  return none;
}

//setup things close to nodes. copied from UTOnslaughtGame.
function FindCloseActors()
{
	local Actor A;
  local int i;
	local UTOnslaughtNodeObjective node;

  for (i = 0; i < ObjectiveInfo.Length; i++)
  {   
        node = UTOnslaughtNodeObjective(ObjectiveInfo[i].Objective);
        
        if (node != none)
        {
             node.InitCloseActors();
        }
   }
   
	ForEach AllActors(class 'Actor', A)
	{
		if (UTWarfarePlayerStart(A) != None)
		{
			node = ClosestNodeObjectiveTo(A);
			if (node != none)
			{
			   node.PlayerStarts[node.PlayerStarts.Length] = PlayerStart(A);
				 node.bIsTeleportDestination = true;
      }
		}
		else if ( UTVehicleFactory(A) != None )
		{							    
      if (UTVehicleFactory(A).ONSObjectiveOverride != none)
      {			    
			    UTVehicleFactory(A).AddToClosestObjective();
			    UTVehicleFactory(A).Deactivate();
			}
		}
		else if (UTOnslaughtNodeEnhancement(A) != None)
		{
			if (UTOnslaughtNodeEnhancement(A).ControllingNode == None)
			{
				UTOnslaughtNodeEnhancement(A).SetControllingNode(ClosestNodeObjectiveTo(A));
			}
		}
		else if(UTDeployableNodeLocker(A) != none)
		{
			UTDeployableNodeLocker(A).AddToClosestObjective();
		}
		else if (UTOnslaughtNodeTeleporter(A) != None)
		{
			Node = ClosestNodeObjectiveTo(A);
			Node.NodeTeleporters[Node.NodeTeleporters.length] = UTOnslaughtNodeTeleporter(A);
		}
	}
}

function RestartBRGame()
{
    bGameRestarted = false;
    EndTime = 0;
 		bAlreadyChanged = true;
    RestartGame();
}

//apply settings to the game
function ApplySettings()
{
    local int i;
    local bool found;
    local UTPawn P;
    local UTWeap_Translocator tl;
    local UTBRPlayerManager pm;

    if (Role < ROLE_Authority)
    {
        return;
    }

    ReadGameSettings();

    if (useBRGravity && (! bool(Settings.StandardUTFeel)))
    {
       WorldInfo.WorldGravityZ = float(Settings.WorldGravityZ);
    }
    
    bAllowTranslocator = bool(Settings.AllowTranslocators);
    bAllowHoverboard = bool(Settings.AllowHoverboards); 

    ForEach WorldInfo.AllActors(class'UTPawn', P)
    {      
        AssignHoverboard(P);
    }
             
    found = false;
    
    for (i = 0; i < DefaultInventory.Length; i++)
    {
        if (DefaultInventory[i] == TranslocatorClass)
        {
            found = true;
            if (! bAllowTranslocator)
            {
                DefaultInventory.Remove(i, 1);
            }
            break;
        }
    }
    
    if ((! found) && bAllowTranslocator)
    {
        DefaultInventory[DefaultInventory.Length] = TranslocatorClass;
    }
    
    if (! bAllowTranslocator)
    {
        ForEach WorldInfo.AllActors(class'UTWeap_Translocator', tl)
        {
            tl.destroy();
        }    
    }    
    
    ForEach WorldInfo.AllActors(class'UTPawn', P)
    {
        AssignJumps(P);
    }
    
    SetDodgeJumpTimer();
    
    if (useBRGameSpeed)
    {
	    WorldInfo.Game.SetGameSpeed(float(Settings.GameSpeedMultiplier));
	  }
	  
    ApplySettingsToBall();
	  
    ForEach WorldInfo.AllActors(class'UTBRPlayerManager', pm)
    {
        pm.UpdateClientGameSettings(Settings.Serialize());
    }
}

simulated function ApplySettingsToBall()
{
    local UTBRBall ball;
    
    ForEach WorldInfo.AllActors(class'UTBRBall', ball)
    {
        ball.SetBallMaterials();
    }
}

simulated function ClientApplySettings()
{   
    SetDodgeJumpTimer();   
    ApplySettingsToBall();
}

simulated function SetDodgeJumpTimer()
{
    if (bool(Settings.AllowDodgeJumping) && (! bool(Settings.StandardUTFeel)))
    {
        //must run on both client and server. if not on client then it won't work at all.
        //if not on server jumping is jerky.
        SetTimer(0.25, true, 'DodgeJumpTimer');
    }
    else
    {
        ClearTimer('DodgeJumpTimer');
    }
}

//return how high a1 is over a2, negative if below. 
function float VerticalDistanceBetween(Actor a1, Actor a2)
{
    if ((a1 == none) || (a2 == none))
    {
        return 0;
    }
    
    return a1.Location.Z - a2.Location.Z;
}

//return the horizontal distance a1 is away from a2.
//this is the distance from z to z along x/y plane.
function float HorizontalDistanceBetween(Actor a1, Actor a2)
{   
    if ((a1 == none) || (a2 == none))
    {
        return 0;
    }
    
    return HorizontalDistanceBetweenPoints(a1.location, a2.location);  
}

//return the horizontal distance v1 is away from v2.
//this is the distance from z to z along x/y plane.
function float HorizontalDistanceBetweenPoints(vector v1, vector v2)
{      
    v1.Z = v2.Z;
    return Vsize(v1 - v2);
}

//return how far away a1 is from a2
function float DistanceBetween(Actor a1, Actor a2)
{
  if ((a1 == none) || (a2 == none))
  {
      return 0;
  }
  
  return DistanceBetweenPoints(a1.Location, a2.Location);
}

//return how far away a1 is from a2
function float DistanceBetweenPoints(vector v1, vector v2)
{  
    return vsize(v1 - v2);
}

function bool IsLowGravity()
{
    //normal grav is -520. lowgrav mutator is -100. 
    return WorldInfo.WorldGravityZ > -300;
}

//return multiplier according to gravity for travel distance calculations
//for reg grav this is 1 and for low grav this is 2.
function float GetTravelGravityMultiplier()
{
    local float gravDiff, gravMult;
    
    //things travel about twice as far in low grav.
    //diff between -420(regular br grav) and -100(low grav) is 320 at which we have a multiplier of 2.
        
    gravDiff = (-420 - WorldInfo.WorldGravityZ) * -1;
    gravMult = 2;
    gravMult = (1 + ((gravDiff / 320) * (gravMult - 1)));
    
    return gravMult;
}

event InitGame( string Options, out string ErrorMessage )
{
    local string mutName, s;
    local UTUIDataProvider_GameModeInfo gmi;
  
  	s = ParseOption(Options, "AdminRightsNotNeeded");
  	if (s != "")
  	{
  	    AdminRightsNotNeeded = bool(s);
  	}
  	   
    gmi = new (none, "UTBRGame") class'BombingRun.UTUIDataProvider_GameModeInfo';
             
    gmi.Description = "Both teams seek possesion of a single Orb. Orb Carrier must take the orb to the enemy goal to score. Author: Mark Caldwell Aka W@rHe@d of The Reliquary http://www.relicsut.com.";
    gmi.GameMode = "BombingRun.UTBRGame";
    gmi.GameSettingsClass="BombingRun.UTGameSettingsBR";
    gmi.GameSearchClass="UTGameSearchCustom";
    gmi.OptionSet="VCTF";
    gmi.FriendlyName="BombingRun";
    //gmi.PreviewImageMarkup="<Images:UI_FrontEnd_Art.GameTypes.TeamDeathmatch>";
    gmi.PreviewImageMarkup="<Images:BombingRunGraphics.GameTypePicture>";    
    gmi.Prefixes="BR";
    gmi.IconImage="UI_HUD.HUD.UI_HUD_BaseD";
    gmi.ModClientSettingsScene = "BombingRunGraphics.BRGameSettings";
    gmi.ModGameSettingsScene = "BombingRunGraphics.BRGameSettings";

    gmi.IconU=230;
    gmi.IconV=76;
    gmi.IconUL=104;
    gmi.IconVL=113;
   
    gmi.SaveConfig();

    ReadGameSettings();         
          
    s = Settings.Mutators;
    
    while (true)
    {
       mutName = Parse(s, ","); 
       if (mutName == "") 
       {
           break;
       }
        
       AddMutator(mutName, true);       
    }
         
    super.InitGame(Options, ErrorMessage);
            
    WarmupRemaining = WarmupTime;
    bWarmupRound = (WarmupTime > 0);
    
    GameEnded = false;  
    InitedManagers = false;
    PlayerManagerArray.Length = 0;    
     
    SetDebugStr(Settings.Debug);
    
    //this is a UT3 patch 2.0 bug. need to bump up by one to have correct selected difficulty. does not happen in online play.
    if (WorldInfo.Netmode==NM_Standalone)
    {
        GameDifficulty = GameDifficulty + 1;
        AdjustedDifficulty = GameDifficulty;
    }    
    
    SetSkillDescription(); 
}

function ReadGameSettings()
{
    local UTBRSettings mapSettings;
    
    Settings = new (none, "Bombing Run Settings") class'BombingRun.UTBRSettings';
    Settings.SetDefaults();
    Settings.SaveConfig();
          
    mapSettings = new (none, WorldInfo.GetMapName(true)) class'BombingRun.UTBRSettings';
  
    Settings.MergeSettings(mapSettings);
}

function UpdateGameSettings()
{
	local UTMutator M;
	local string mutatorList;
	
  //update online server browser mutator list
	RemoveOption(ServerOptions, "Mutator");

    mutatorList = "";
	M = GetBaseUTMutator();
	while (M != none )
	{
		mutatorList $= mutatorList == "" ? Pathname(M.Class) : (","$Pathname(M.Class));
		M = M.GetNextUTMutator();
	}
    	
	if (mutatorList != "")
	{
	    if (ServerOptions != "")
        {
            ServerOptions $= "?";
        }
	    ServerOptions $= "Mutator=" $ mutatorList;
	}
	
	Super.UpdateGameSettings();
}


/*
copied from UTTeamGame and modified for br.
in BR we always allow a player to change teams. teams get out of balance as players come and go
and players must be able to change to correct the teams. 
also, team balance is not just determined by number of players but also balance of power.
a team with less powerful players should be able to have more players on it if people want it that way.
even if just for fun, people should be able to setup their match the way they want and not have the
game force the teams on them.
granted this feature can be abused but better that than not being able to change teams.
*/
/** ChangeTeam()
* verify whether controller Other is allowed to change team, and if so change his team by calling SetTeam().
* @param Other:  the controller which wants to change teams
* @param num:  the teamindex of the desired team.  If 255, pick the smallest team.
* @param bNewTeam:  if true, broadcast team change notification
*/
function bool ChangeTeam(Controller Other, int num, bool bNewTeam)
{
	local UTTeamInfo NewTeam;

	// no team changes after initial change if single player campaign
	if ( (SinglePlayerMissionID != -1) && (Other.PlayerReplicationInfo.Team != None) )
	{
		return false;
	}

	// check if only allow team changes before match starts
	if ( bMustJoinBeforeStart && GameReplicationInfo.bMatchHasBegun )
		return false;

	// don't add spectators to teams
	if ( Other.IsA('PlayerController') && Other.PlayerReplicationInfo.bOnlySpectator )
	{
		Other.PlayerReplicationInfo.Team = None;
		return true;
	}

    //simple change for br to allow players to always switch
    NewTeam = Teams[num];
    
    /*
	NewTeam = (num < 255) ? Teams[PickTeam(num,Other)] : None;

	// check if already on this team
	if ( Other.PlayerReplicationInfo.Team == NewTeam )
	{
		// may have returned current team if not allowed to switch
		// if not allowed to switch, set up or complete proposed swap
		if ( (num < 2) && (num != NewTeam.TeamIndex) && !bPlayersVsBots && (PlayerController(Other) != None)
			&& (UTTeamInfo(Other.PlayerReplicationInfo.Team) != None) )
		{
			// check if swap request is pending
			if ( PendingTeamSwap != None )
			{
				if ( PendingTeamSwap.bDeleteMe || (WorldInfo.TimeSeconds - SwapRequestTime > 8.0) )
				{
					PendingTeamSwap = None;
				}
				else if ( PendingTeamSwap.PlayerReplicationInfo.Team.TeamIndex == num )
				{
					// do the swap!
					PendingTeam = UTTeamInfo(PendingTeamSwap.PlayerReplicationInfo.Team);
					if (PendingTeam != None )
					{
						SetTeam(PendingTeamSwap, UTTeamInfo(Other.PlayerReplicationInfo.Team), true);
						if ( PendingTeamSwap.Pawn != None )
							PendingTeamSwap.Pawn.PlayerChangedTeam();
						PendingTeamSwap = None;
						SetTeam(Other, PendingTeam, bNewTeam);
						return true;
					}
				}
			}

			// set pending swap request
			PendingTeamSwap = PlayerController(Other);
			SwapRequestTime = WorldInfo.TimeSeconds;

			// broadcast swap request
			BroadcastLocalizedMessage(class'UTTeamGameMessage', 0, PendingTeamSwap.PlayerReplicationInfo);
		}
		return false;
	}
	*/

	// set the new team for Other
	SetTeam(Other, NewTeam, bNewTeam);
	return true;
}

function SetDebugStr(string debugStr)
{  
   local string oldDebug;
   
   oldDebug = Settings.Debug;
   
   Settings.Debug = debugStr;

   if (oldDebug != "")
   {
       SendDebugMessage("Stopping all debugging");
   }
      
   DebugMatchStartup = (InStr(CAPS(Settings.Debug), "MATCHSTARTUP") >= 0);
    
   if (DebugMatchStartup)
   {
       SendDebugMessage("Match startup debug logging started");       
   }

   DebugHud = (InStr(CAPS(Settings.Debug), "HUD") >= 0);
      
   if (DebugHud)
   {
       SendDebugMessage("Hud debug logging started");       
   }
  
}

simulated function bool MatchIsInProgress()
{
    //needed because GameReplicationInfo.Timer on client side checks this function or WorldInfo.Game=none.
    //because for br we replication this game class we must rig this function to always return true client side. 
	if (WorldInfo.NetMode == NM_Client)
	{
	    return true;
	}
	
	return false;
}
	
simulated event ReplicatedEvent(name VarName)
{
    if (VarName == 'DebugMatchStartup') 
    {
        if (DebugMatchStartup)
        {
            SetTimer(1.0, true, 'DebugMatchStartupTimer');
        }
        else
        {
            ClearTimer('DebugMatchStartupTimer');        
        }
    }
    
    if (VarName == 'SkillDescription') 
    {
        default.SkillDescription = SkillDescription;
    }    
}

//spit out playercontroller debug info on client
simulated function DebugMatchStartupTimer()
{
    local PlayerController pc;
        
    foreach WorldInfo.AllControllers(class'PlayerController', pc)
    {
        SendDebugMessage("Client debug for " $ pc.PlayerReplicationInfo.GetPlayerAlias() $ "  state=" $ pc.GetStateName());
    } 
}

static function string Parse(out string source, string delimiter)
{
  local int i;
  local string ret;

  i = inStr(source, delimiter);
  if (i >= 0)
  {
    ret = Mid(source, 0, i);
    source = Mid(source, i + Len(delimiter));
  }
  else
  {
    ret = source;
    source = "";
  }

  return ret;
}

static function string Trim(string s)
{
  while (Mid(s, 0, 1) == " ")
  {
    s = Mid(s, 1);
  }

  while (Mid(s, Len(s) - 1, 1) == " ")
  {
    s = Mid(s, 0, Len(s) - 1);
  }

  return s;
}

function UTBRObjectiveInfo GetObjectiveInfo(Actor objective, optional UTBRPlayerManager playerManager)
{
    local int i;
      
    for (i = 0; i < ObjectiveInfo.Length; i++)
    {    
        if (ObjectiveInfo[i].Objective == objective)
        {
            return ObjectiveInfo[i]; 
        } 
    }
    
    return none;
}

//return perpendicular point dist away from goal in direction of 1 or -1
function Vector PointAwayFromGoal(Actor goal, float dist, int direction)
{
    local Rotator r;
    local vector upUnit;

    //a unit vector of dist length along z and then rolled is perpendicular to goal face 
    //if goal has no rotation. then we rotate that vector using goal's rotation to get 
    //the vector properly aligned to goal face.
        
    upUnit.z = dist;  
    r.roll = 16384 * direction;
    
    return goal.Location + ((upUnit >> r) >> goal.Rotation);
}

function AddObjectiveInfo(Actor objective)
{
    local int i, cnt;
    local UTBRPathNode brNode;
    local UTBRObjectiveInfo info;
    local float best, dist;
    local UTKillZVolume killZone;
    local vector temp1, temp2;
	  local UTOnslaughtNodeObjective node;
	  local UTDefensePoint dp;
       

    info = Spawn(class'UTBRObjectiveInfo');
    
    info.Objective = objective;
    
  	foreach AllActors(class'UTBRPathNode', brNode)
	{
	    for (i = 0; i < brNode.RouteInfo.Length; i++)
	    {
	        if (brNode.RouteInfo[i].RouteTarget == objective)
	        {
                info.BRPathNodes[info.BRPathNodes.Length] = brNode;
                                               	
	            break;
            }
        }   
	}
	
	//this routine must be called from postbeginplay, not initgame, or errors will occur
	//and node will not be inited properly
	
    node = UTOnslaughtNodeObjective(objective);
    if (node != none)
    {
        Node.bStandalone = true;	
	      Node.SetInitialState();	      
    }

    if (UTBRGoal(objective) != none)
    {
        //find jump pad pointing to goal
        info.JumpPad = FindGoalJumpPad(UTBRGoal(objective));

        //find kilzone to reset ball to          
        best = 9999; 
        ForEach WorldInfo.AllActors(class'UTKillZVolume', killZone)
        {
            dist = DistanceBetween(objective, killZone);
            if ((dist < 2000) && (dist < best))
            {
                best = dist;
                info.KillZone = killZone;
            }    
        }
        
        //find what bot should aim at to reset ball.
        //for killzones behind the goal, it's best to aim at the goal
        //because the killzone center could be far behind goal which would
        //make bot aim too low and hit the goal rim.
        if (info.KillZone != none)
        {
            info.BallResetTarget = info.KillZone;
            temp1 = PointAwayFromGoal(UTBRGoal(objective), HorizontalDistanceBetween(info.KillZone, UTBRGoal(objective)), 1);
            temp2 = PointAwayFromGoal(UTBRGoal(objective), HorizontalDistanceBetween(info.KillZone, UTBRGoal(objective)), -1);
            
            //want to check behind goal so use vector which is furthest away from ball base
            if (HorizontalDistanceBetweenPoints(TheBall.HomeBase.Location, temp2) > 
                 HorizontalDistanceBetweenPoints(TheBall.HomeBase.Location, temp1))
            {
                temp1 = temp2;
            }
            
            if (DistanceBetweenPoints(temp1, info.KillZone.Location) <= 100)
            {
                info.BallResetTarget = objective;
            }
        }                
        
        cnt = 0;
        foreach WorldInfo.RadiusNavigationPoints(class'UTDefensePoint', dp, objective.Location, 5000)
        {
           cnt++;
        }
        
        if (cnt >= 7)
        {
            info.HasEnoughDefensePoints = true;
        }
    }  
    	
	ObjectiveInfo[ObjectiveInfo.Length] = info;
}

function SetupObjectiveInfo()
{
    local UTOnslaughtNodeObjective powerNode;
    local UTBRGoal goal;
    local UTBRBallBase ballBase;
    local UTBRPathNode brNode;
    local Actor obj;
    local int i;
       
  	foreach AllActors(class'UTOnslaughtNodeObjective', powerNode)
	{
	    AddObjectiveInfo(powerNode);
	}
	
  	foreach AllActors(class'UTBRGoal', goal)
	{
	    AddObjectiveInfo(goal);
	}
    
  	foreach AllActors(class'UTBRBallBase', ballBase)
	{
	    AddObjectiveInfo(ballBase);
	}
    
    //look through each routeTarget in all brpathnodes and add any needed
    //objective info. this allows mappers to set paths to any actor such as
    //pickups and vehicles.
  	foreach AllActors(class'UTBRPathNode', brNode)
	{
	    for (i = 0; i < brNode.RouteInfo.Length; i++)
	    {
	        obj = brNode.RouteInfo[i].RouteTarget;
	        if ((obj != none) && (GetObjectiveInfo(obj) == none))
            {
                AddObjectiveInfo(obj);
            }
        }
	}    	    	
}

function UTJumpPad FindGoalJumpPad(UTBRGoal goal)
{
    local UTJumpPad ret;
    local UTJumpPad jp;    
    local float jBest, dist;
    local vector target;
    
    //find jumppad which can be used to score with
    jBest = 999999;    
    foreach WorldInfo.RadiusNavigationPoints(class'UTJumpPad', jp, goal.Location, 3000)
    {        
        if (jp.JumpTarget == goal)
        {
            ret = jp;
            break;
        }
       
        if (jp.JumpTarget == none)
        {
            continue;
        }
        
        //if pad can't get bot near goal, then don't use it
        if ((DistanceBetween(jp, goal) - DistanceBetween(jp.JumpTarget, jp)) > 1000)
        {
           continue;
        }
                       
        target = jp.Location + (Normal(jp.JumpTarget.Location - jp.Location) * DistanceBetween(jp, goal));
            
        //if jumppad takes bot closer to goal, or jump target is close to goal, then use it                   
        if (
            (vsize(goal.Location - target) < DistanceBetween(jp, goal)) ||
            (DistanceBetween(jp.JumpTarget, goal) < 1000) 
           )
        {
            dist = vsize(goal.Location - target);
            
            if (dist < jBest)
            {   
                ret = jp;
                jBest = dist;               
            }
        }  
    }
    
    return ret;
}

simulated function UTBRPlayerManager GetPlayerManager(PlayerReplicationInfo PRI)
{
    local int i;
      
    for (i = 0; i < PlayerManagerArray.Length; i++)
    {
        if (PlayerManagerArray[i] == none)
        {
            continue;
        }
        
        if (PlayerManagerArray[i].PlayerReplicationInfo == PRI)
        {
            return PlayerManagerArray[i]; 
        }
    }
    
    return none;
}

function UTBRPlayerManager NewPlayerManager(Controller C)
{
    local UTBRPlayerManager playerManager;

    playerManager = spawn(class'UTBRPlayerManager');
    playerManager.SetOwner(C);
    playerManager.PlayerReplicationInfo = C.PlayerReplicationInfo;
    playerManager.Controller = C;
    playerManager.Bot = UTBot(C);
    PlayerManagerArray[PlayerManagerArray.Length] = playerManager;
       
    return PlayerManager; 
}

//send a debug message to other player's console
simulated function SendDebugMessage(string message)
{
    local Controller C;
    local string s, s2;
    local bool first;
        
    foreach WorldInfo.AllControllers(class'Controller', C)
    {
        if (PlayerController(C) != none)
        {
            s = message;
            first = true;
            while (s != "")
            {    
                s2 = mid(s, 0, 120);
                if (! first)
                {
                    s2 = "  " $ s2;
                }
                s = mid(s, 120);
                PlayerController(C).ClientMessage(s2);
                first = false;
            }
        }
    }
    
    LogInternal(message);       
}

/**
 * returns true if Viewer is allowed to spectate ViewTarget
 **/
function bool CanSpectate( PlayerController Viewer, PlayerReplicationInfo ViewTarget )
{
    if ( (ViewTarget == None) || ViewTarget.bOnlySpectator )
        return false;
                
    return ( Viewer.PlayerReplicationInfo.bOnlySpectator || (ViewTarget.Team == Viewer.PlayerReplicationInfo.Team) );
}

//set players to be allowed to jump at apex of a dodge like in ut2k4/ut2k3
simulated function DodgeJumpTimer()
{
    local PlayerController pc;
        
    foreach WorldInfo.AllControllers(class'PlayerController', pc)
    {
        if ((UTPawn(pc.Pawn) != none) && (UTPawn(pc.Pawn).bDodging))
        {
            UTPawn(pc.Pawn).bReadyToDoubleJump = true; 
        }
    }   
}

auto State PendingMatch
{   
	function BeginState(Name PreviousStateName)
	{
	    Super.BeginState(PreviousStateName);
        IsPendingMatch = true;		    
	}

	function EndState(Name NextStateName)
	{
	    Super.EndState(NextStateName);
        IsPendingMatch = false;		    
	}
	
    simulated function bool MatchIsInProgress()
    {
        return Global.MatchIsInProgress();
    }
    
    //allow a single spectator to start bot match
	function Timer()
	{
		local PlayerController P;
		local bool bReady;
		local string s;
		
		Super.Timer();
		
		CheckVoteTime();
			
		if (! InitedManagers)
		{
            //must init spectators at start of new match because they don't relogin but stay logged in.
            //not doing this will result a spectator losing their br enhanced camera on 2nd match.
            //tried doing this in InitGame but did not work there.
            //also must init human players still present from last match so that an admin can run admin commands.
            
            InitedManagers = true;
            
        	foreach WorldInfo.AllControllers(class'PlayerController', P)
        	{    
        		if ((P.PlayerReplicationInfo != None) && P.bIsPlayer)
        		{
                    if (GetPlayerManager(P.PlayerReplicationInfo) == none)
                    {
                        NewPlayerManager(P);
                    }		
        		}
        	}  		
        }
		
		if (DebugMatchStartup)
		{
 	        LogInternal("Start MatchStartup Debug Info");		
 	        LogInternal(
 	          "bStartedCountDown=" $ bStartedCountDown $
 	          ",bPlayersMustBeReady=" $ bPlayersMustBeReady $
 	          ",bWaitForNetPlayers=" $ bWaitForNetPlayers $
 	          ",NumPlayers=" $ NumPlayers $
 	          ",MinNetPlayers=" $ MinNetPlayers $
 	          ",PendingMatchElapsedTime=" $ PendingMatchElapsedTime $
 	          ",MaxPlayers=" $ MaxPlayers $
 	          ",NetWait=" $ NetWait $
 	          ",ClientProcessingTimeout=" $ ClientProcessingTimeout $
 	          ",InitialProcessingIsComplete()=" $ InitialProcessingIsComplete() $
 	          ",CountDown=" $ CountDown $ 
 	          ",TimeSeconds=" $ WorldInfo.TimeSeconds $
 	          ",NetMode=" $ WorldInfo.NetMode $
 	          ",bWaitingToStartMatch=" $ bWaitingToStartMatch $
 	          ",bWarmupRound=" $ bWarmupRound $
 	          ",WarmupTime=" $ WarmupTime $
 	          ",State=" $ GetStateName()
             );
             
			foreach WorldInfo.AllControllers(class'PlayerController', P)
			{
			    if (UTPlayerController(P) != none)
			    {
			        s = ",bInitialProcessingComplete=" $ UTPlayerController(P).bInitialProcessingComplete;
                }
                
			    LogInternal(
                 "Player=" $ P.PlayerreplicationInfo.GetPlayerAlias() $ 
                 ",HasClientLoadedCurrentWorld()=" $ P.HasClientLoadedCurrentWorld() $
                 ",bReadyToPlay=" $ P.PlayerReplicationInfo.bReadyToPlay $
                 ",bWaitingPlayer=" $ P.PlayerReplicationInfo.bWaitingPlayer $
                 ",bOnlySpectator=" $ P.PlayerReplicationInfo.bOnlySpectator $
                 ",bIsSpectator=" $ P.PlayerReplicationInfo.bIsSpectator $
                 ",bIsPlayer=" $ P.bIsPlayer $ 
                 ",State=" $ P.GetStateName() $
                 ",WaitDelay=" $ P.WaitDelay $
                 ",PlayerController Owner=" $ P.Owner $
                 s
                 );			    
            }
            
 	        LogInternal("End MatchStartup Debug Info");            		
        }
		
		if (NumPlayers > 0)
		{
		    return;
        }

        //no players online, but check and see if spectator is waiting
        
		bReady = false;

		if (!bStartedCountDown)
		{
			foreach WorldInfo.AllControllers(class'PlayerController', P)
			{
			    //note: WebAdmin adds non player spectators to game so P.bIsPlayer test filters them out
			    
				if ((P.PlayerReplicationInfo != None) && (P.PlayerReplicationInfo.bOnlySpectator) && P.bIsPlayer)
				{
				    LogInternal("Starting match because a spectator(" $ P.PlayerreplicationInfo.GetPlayerAlias() $ ") has joined and no other players have joined.");
					bReady = true;
					break;
				}
			}	
		}
		
		if (bReady)
		{
			bStartedCountDown = true;
			bWarmupRound = false;
			
			CountDown = savCountDown - 1;
			
			if ( CountDown <= 0 )
				StartMatch();
			else
				StartupStage = 5 - CountDown;
				
            savCountDown = CountDown;			
		}			
	}
}

function ResetLevel()
{
    TheBall.SetEnabled(false);
    TheBall.SendHome(none);
        
    Super.ResetLevel();
}

function StartMatch()
{
    super.StartMatch();
       
    TheBall.SetEnabled(true);
    
	if (DebugMatchStartup)
	{
	    LogInternal("MatchStartup Debug Info. StartMatch called. Stack Trace:");
        ScriptTrace();
    }
}


/* CheckRelevance()
returns true if actor is relevant to this game and should not be destroyed.  Called in Actor.PreBeginPlay(), intended to allow
mutators to remove or replace actors being spawned
*/
function bool CheckRelevance(Actor Other)
{
    if (bool(Settings.CarryBallInVehicles) && other.isa('UTVehicle'))
    {
        UTVehicle(other).bCanCarryFlag = true;
    }

    if (
        (! bool(Settings.AllowVehicles)) && (! other.isa('UTVehicle_Hoverboard')) &&
        (other.isa('UTVehicleFactory') || other.isa('UTVehicle'))
       )
    {
        return false;
    }
    
    return super.CheckRelevance(Other);
}

// Monitor killed messages for fraglimit
function Killed( Controller Killer, Controller KilledPlayer, Pawn KilledPawn, class<DamageType> damageType )
{
    super.Killed(Killer, KilledPlayer, KilledPawn, damageType);
   
    if (TheBall.LastHolder == KilledPawn)
    {
        TheBall.LastHolder = none;
    }
    
    if (WorldInfo.Game.bGameEnded && (EndGameFocus == KilledPawn))
    {
       SetEndGameFocus(none);
    }     
}

function bool PreventDeath(Pawn KilledPawn, Controller Killer, class<DamageType> damageType, vector HitLocation)
{
    //needed when restarting round and two players spawn at same place. 
    //happens when not enough start spots are in map.
    if (SpawnProtecting && (damageType == class'UTDmgType_Encroached'))
    {
        return true;
    }

    return super.PreventDeath(KilledPawn, Killer, damageType, HitLocation);
}

event PostLogin( playercontroller NewPlayer )
{
    super.PostLogin(NewPlayer);

    //needed for spectators
    if (GetPlayerManager(NewPlayer.PlayerReplicationInfo) == none)
    {
       NewPlayerManager(NewPlayer);
       
       //this will only work here for spectators.
       //regular players have this done in RestartPlayer.
       NewPlayer.ClientMessage(GetVersionMessage());          
    }    
}

//called to spawn player 
function RestartPlayer(Controller aPlayer)
{
    local UTBRPlayerManager playerManager;
    local UTBRBallLauncher bl;
    
    Super.RestartPlayer(aPlayer);

    playerManager = GetPlayerManager(aPlayer.PlayerReplicationInfo);
    if (playerManager == none)
    {
        playerManager = NewPlayerManager(aPlayer);
       
        //dislay version message when player first spawns into game, then
        //can't do this in PostLogin because it's still too early in the process and
        //the message won't show up in net play.
        if (PlayerController(aplayer) != none)
        {
            PlayerController(aplayer).ClientMessage(GetVersionMessage());          
        }       
    }
          
    bl = Spawn(class'UTBRBallLauncher', aPlayer.Pawn);
    
    playerManager.BallLauncher = bl;
    bl.PlayerManager = playerManager;
    bl.TeamIndex = aPlayer.GetTeamNum();
    
    /*
    there is bug in ut3 code in which client weapon state does not get set to inactive and remains in
    state PendingClientWeaponSet when bNeverSwitchOnPickup is true.
    bug is in InventoryManager.ClientWeaponSet().
    this causes the Zoom on sniper weapons not to work.
    the bug is triggered when you call GiveTo() but not when you call AddInventory(w, true), where
    true is to not activate weapon.
    
    bl.GiveTo(aPlayer.Pawn);
    */
    if (aPlayer.Pawn != none)
    {
        aPlayer.Pawn.InvManager.AddInventory(bl, true);
    }
       
    InitOnRespawn(playerManager);
    playerManager.UseBRMultiJump = UTPawn(aPlayer.Pawn).MaxMultiJump == class'UTPawn'.Default.MaxMultiJump;
    playerManager.UseBRMultiJumpBoost = UTPawn(aPlayer.Pawn).MultiJumpBoost == class'UTPawn'.Default.MultiJumpBoost;
    
    AssignJumps(UTPawn(aPlayer.Pawn));
}

function AssignJumps(UTPawn P)
{
    local UTBRPlayerManager pm;
    
    if ((P == none) || (P.Controller == none))
    {
        return;
    }
    
    pm = GetPlayerManager(P.Controller.PlayerReplicationInfo);
    
    if (pm == none)
    {
        return;
    }
    
    if (! bool(Settings.StandardUTFeel))
    {
        if (pm.UseBRMultiJumpBoost)
        {
            P.MultiJumpBoost = int(Settings.MultiJumpBoost);
        }
    
        if (pm.UseBRMultiJump)
		    {
            P.MaxMultiJump = int(Settings.MaxMultiJump);
		        P.MultiJumpRemaining = int(Settings.MaxMultiJump);                      		
        }
    }	
}

function BRGameTimer()
{
    local PlayerController PC;
    local UTBot B;
    local UTVehicle v;
    local UTOnslaughtPowerNode node;
    local UTDeployablePickupFactory dep;    


    if (roundTimer > 0)
    {
        roundTimer = roundTimer - 1;
        
        if (roundTimer == 0)
        {
            RestartPlayers();
            TheBall.SetEnabled(true);
            SetPlayerObjectives(true);          
        }
        else if (roundTimer <= 10)
        {
            foreach WorldInfo.AllControllers(class'PlayerController', PC)
            {
                PC.ReceiveLocalizedMessage( class'UTTimerMessage', roundTimer );                
            }
        }   
    }           
    
    if (Test_NoWeaponVolumes)
    {
        foreach WorldInfo.AllActors(class'UTDeployablePickupFactory', dep)
        {
            //this does not work
            //dep.destroy();
            
            dep.DisablePickup();            
        }   
    }
    
    if (Test_HealthyBots)
    {
        foreach WorldInfo.AllControllers(class'UTBot', B)
        {
            B.Pawn.Health = 100;
        }    
    }
    
    if (Test_DestroyVehicles)
    {
        ForEach WorldInfo.AllActors(class'UTVehicle', v)
        {
           if ((UTVehicle_Hoverboard(v) == none) && (V != TestBotVehicle))
           {
               v.destroy();
               v.Health = 0;               
           }
        }    
    }
    
    //must do this or nodes gets confused and think an opposing team orb is it's own and so
    //it would not rebuild node      
    foreach AllActors(class'UTOnslaughtPowerNode', node)
    {        
        node.ControllingFlag = none;
        
        //code in node class tests for onslaught pri so it fails 
        //so we award some points here
        if (node.IsNeutral() && (node.LastDamagedBy != none))
        {
            node.LastDamagedBy.Score += 2;
            node.LastDamagedBy = none;
        }
    }   
}

//initiate new round sequence after a goal score is made
function StartNewRound()
{
    roundTimer = 14;
}

//called during new round sequence to start all players at spawn points
function RestartPlayers()
{
    local Controller C;
    local TeamStartsStruct teamStarts[2];
    local PlayerStart ps;
    local int ix, tn, ix2, ix3, teamNum;
    local UTProj_TransDisc disc;
    local UTBRPlayerManager playerManager;
    local UTOnslaughtNodeObjective node;    

    foreach WorldInfo.AllActors (class'UTProj_TransDisc', disc)
    {
        disc.destroy();
    }  

    
    foreach WorldInfo.AllNavigationPoints(class'PlayerStart', ps)
    {     	  
        teamNum = -1;
        
        if (UTWarfarePlayerStart(ps) != none)
        {
            node = FindParentNode(ps);
                    
            if (node != none)
            {
                teamNum = node.DefenderTeamIndex;
                if (! NodeActivated(node))
                {
                    continue;
                }            
            }
        }
        
    	  if (UTTeamPlayerStart(ps) != none)
    	  {
    	      teamNum = UTTeamPlayerStart(ps).TeamNumber;
    	  }   

        if (teamNum >= 0)
        {
            teamStarts[teamNum].StartPoints[teamStarts[teamNum].StartPoints.length] = ps;
        }
        else
        {
            teamStarts[0].StartPoints[teamStarts[0].StartPoints.length] = ps;
            teamStarts[1].StartPoints[teamStarts[1].StartPoints.length] = ps;                
        }
    }
    
    //scramble start points randomly so players spawn at different places each time
    for (ix = 0; ix < 2; ix++)
    {
        for (ix2 = 0; ix2 < teamStarts[ix].StartPoints.length; ix2++)
        {
            ix3 = rand(teamStarts[ix].StartPoints.length);
            ps = teamStarts[ix].StartPoints[ix2];
            teamStarts[ix].StartPoints[ix2] = teamStarts[ix].StartPoints[ix3];
            teamStarts[ix].StartPoints[ix3] = ps;
        }
    }
  
    foreach WorldInfo.AllControllers(class'Controller', C)
    {
        if (vehicle(c.Pawn) != none)
        {
            vehicle(c.Pawn).DriverLeave(true);
        }
    
		if (C.PlayerReplicationInfo != None && !C.PlayerReplicationInfo.bOnlySpectator)       
        {
            if (C.Pawn == none)
            {
                continue;
            }           
            
            tn = C.GetTeamNum();
            ix = teamStarts[tn].ix;
            
            if (ix >= teamStarts[tn].StartPoints.length)
            {
                ix = 0;
            }

            SpawnProtecting = true;
            C.Pawn.Acceleration = vect(0,0,0);
            C.Pawn.Velocity = vect(0,0,0);
            C.Pawn.SetLocation(teamStarts[tn].StartPoints[ix].Location);
            C.SetRotation(teamStarts[tn].StartPoints[ix].Rotation);

            //must kick a feigner out of feign mode or else when they press F to 
            //end feign, they will remain where they were at and not be at the respawn point            
            if ((UTPawn(C.Pawn) != none) && UTPawn(C.Pawn).bFeigningDeath)
            {
               UTPawn(C.Pawn).bFeigningDeath = false;
               UTPawn(C.Pawn).PlayFeignDeath();
            }
            
            SpawnProtecting = false;
           
            ix++;
            teamStarts[tn].ix = ix;            
            
            C.Pawn.PlayTeleportEffect(true, true);

            if (C.Pawn.Health < C.Pawn.HealthMax)
            {   
                C.Pawn.Health = C.Pawn.HealthMax;
            }
            
            playerManager = GetPlayerManager(C.PlayerReplicationInfo);
            if (playermanager != none)
            {
                InitOnRespawn(playerManager);
            }
        }
    }       
}

//perform any need inits after respawn.
//could be after dying or after round end.
function InitOnRespawn(UTBRPlayerManager playerManager)
{
    if (playerManager.Bot != none)
    {    
        //a bot will first go and secure the base before moving onward.
        //this makes bot keep a line of defense while attacking.
        //this usually will result in bot taking the center path of the map rather than wandering off
        //to a side area and leaving the base unprotected.
        playerManager.TakingDefensivePathTime = WorldInfo.TimeSeconds;
        
        playerManager.InitSpecialNavigation(true);
        
        playerManager.DoingBallReset = false;
        playerManager.ShootingGoal = false;      

        //keep bots from killing immediately after spawning. very annoying.
        playerManager.SuppressEnemyTime = WorldInfo.TimeSeconds + 1.5 + RandRange(0, 0.5);
        playerManager.SuppressEnemy = true;
        playerManager.SetBotTacticTimer();
        
        //forces init of formation center info to make bot start fresh
        playerManager.FormationCenter = none;
        playerManager.LastFormationCenter = none;
        
        playerManager.SoakTarget = none;
        playerManager.SpecialAiming = false;
        playerManager.TheBall = TheBall;
        
        if (Test_BotsLoaded)
        {
           playerManager.Bot.Pawn.CreateInventory(class'UTGame.UTWeap_Stinger');
           playerManager.Bot.Pawn.CreateInventory(class'UTGame.UTWeap_ShockRifle');
           playerManager.Bot.Pawn.CreateInventory(class'UTGame.UTWeap_LinkGun');                        
		   UTInventoryManager(playerManager.Bot.Pawn.InvManager).AllAmmo(true);
		   UTInventoryManager(playerManager.Bot.Pawn.InvManager).bInfiniteAmmo = true;
        }         
    }                
}

//show path arrows from P to P's current objective
function ShowPathTo(PlayerController P, int TeamNum)
{
    local Actor Best;

    Best = TheBall;
    
    if (TheBall.IsHeldByTeamOf(P.Pawn))
    {
        Best = Goals[1 - P.Pawn.GetTeamNum()];
    }
        
    if (Best != None && P.FindPathToward(Best, false) != None)
    {
        Spawn(class'UTWillowWhisp', P,, P.Pawn.Location);
    }
}

function ScoreKill(Controller Killer, Controller Other)
{
    if (TheBall != none) 
    {
        if (Other.Pawn == TheBall.PassTarget)
        {
            TheBall.ReleasePassTarget();
        }
    }
    
    super(UTTeamGame).ScoreKill(Killer, Other);
}

/** return a value based on how much this pawn needs help */
function int GetHandicapNeed(Pawn Other)
{
    local PlayerReplicationInfo pri;
    
    pri = GetPlayerReplicationInfo(Other);
    if ( (pri == None) || (pri.Team == None) )
    {
        return 0;
    }

    // base handicap on how far team is behind in powercore health
    return Teams[1 - pri.Team.TeamIndex].Score - pri.Team.Score;
}

/**
 * Returns through outparameters what location message to play
 * Returns true if it returned message information
 */
function bool GetLocationFor(Pawn StatusPawn, out Actor LocationObject, out int MessageIndex, int LocationSpeechOffset)
{
    local UTPickupFactory F;
    local UTGameObjective Best;
    local UTBot B;

    // see if it's a bot heading for an objective or a power up
    B = UTBot(GetController(StatusPawn));
    if ( B != None )
    {
        F = UTPickupFactory(GetController(StatusPawn).RouteGoal);
        if ( (F != None) && F.bHasLocationSpeech )
        {
            MessageIndex = 0;
            LocationObject = F;
            return true;
        }
    }

    if ( (Goals[0] != None) && (Goals[1] != None) )
    {
        Best = (VSizeSq(StatusPawn.Location - Goals[0].Location) < VSizeSq(StatusPawn.Location - Goals[1].Location))
                ? Goals[0]
                : Goals[1];

        MessageIndex = Best.GetLocationMessageIndex(B, StatusPawn);
        LocationObject = Best;
        return true;
    }

    MessageIndex = 10;
    return true;
}

function RegisterBallHomeBase(UTBRBallBase homeBase)
{
    BallHomeBase = homeBase;
}

function RegisterBall(UTBRBall ball)
{
    local int i;
    
    TheBall = ball;
    TheBall.HomeBase = BallHomeBase;

    if (UTBRTeamAI(Teams[i].AI) != none)
    {   
        for ( i=0; i<2; i++ )
        {
            UTBRTeamAI(Teams[i].AI).TheBall = TheBall;
        }
    }   
}

function RegisterGoal(UTBRGoal g, int TeamIndex)
{
    Goals[TeamIndex] = g;
}

function bool NearGoal(Controller C)
{
    local UTGameObjective B;

    B = Goals[1 - C.PlayerReplicationInfo.Team.TeamIndex];
    return (VSize(C.Pawn.Location - B.Location) < 1000);
}

function bool WantFastSpawnFor(AIController B)
{
    return true;
}

//return player with best score/stats on team
function PlayerReplicationInfo GetBestPlayer(int teamNum)
{
    local PlayerReplicationInfo best, pri;
    local Controller P;    
    
    foreach WorldInfo.AllControllers(class'Controller', P)
    {   
        pri = P.PlayerReplicationInfo;
        
        if ((pri != None) && 
            (!pri.bOnlySpectator) &&
            (pri.GetTeamNum() == teamNum))
        {
            if (Controller(pri.Owner).Pawn.health == 0)
            {
                continue;
            }
            
            if (best == none)
            {
                best = pri;
                continue;
            }
            
            if (pri.Score > best.Score)
            {
                best = pri;
                continue;
            }
            
            if (pri.Score < best.Score)
            {
                continue;
            }
            
            if (pri.Kills > best.Kills)
            {
                best = pri;
                continue;
            }
            
            if (pri.Kills < best.Kills)
            {
                continue;
            }               
            
            if (pri.Deaths < best.Score)
            {
                best = pri;
            }
        }
    }       

    return best;
}

//set focus on a player at end of game
//will be the winning team's final scorer, or the highest ranked player on winning team if no
//final score was made to win, or if that player dies (by leaving game, for example)
function SetEndGameFocus(PlayerReplicationInfo Winner)
{
    local Controller P;
    local int winningTeamNum;
    local bool isWinner;

        
    GameEnded = true; 
       
    winningTeamNum = 0;

    if ( Teams[1].Score > Teams[0].Score )
    {
        winningTeamNum = 1;
    }
                        
    //focus on person who scored the winning goal.
    //if they are not found, then focus on the winning team's high scorer
        
    if ((Winner != none) && (Controller(Winner.Owner).Pawn.health == 0))
    {
        Winner = none;
    }               
                
    if (Winner == none)
    {
        Winner = GetBestPlayer(winningTeamNum);
        
        if (Winner == none)
        {
            Winner = GetBestPlayer(1 - winningTeamNum);            
        }
    }
        
    if (Winner != None)
    {
        EndGameFocus = Controller(Winner.Owner).Pawn;

        if (UTVehicle(EndGameFocus) != none)
        {
            //if we allowed focus to be on vehicle then players can't use
            //mouse to rotate their view. some kinda ut glitch in player controller.
            EndGameFocus = UTVehicle(EndGameFocus).Driver; 
        }
    }
        
    if (EndGameFocus == None)
    {
        EndGameFocus = BallHomeBase;
    }
         
	foreach WorldInfo.AllControllers(class'Controller', P)
	{
	    isWinner =
           (Pawn(EndGameFocus) != none) &&
           (Pawn(EndGameFocus).PlayerReplicationInfo != none) && 
           (Pawn(EndGameFocus).PlayerReplicationInfo == P.PlayerReplicationInfo);

        /*
           player rotations were being altered in the end game winner pic from their
           original rotations at end of game. this is a ut bug and happens in other
           game types.
           the fix is on server side to set view target to none.
           not sure why that works but must have something to do with GetPlayerViewPoint() modifying controller rotation.
           the problem is caused because on server side the pawn rotation adjustments
           are not turned off.
           tried overriiding controller rotation but it did not work.
        */
	    if (UTPlayerController(P) != none)
        {         
		    UTPlayerController(P).SetViewTarget(none);
	        P.GotoState('RoundEnded');
	        UTPlayerController(P).ClientGameEnded(EndGameFocus, isWinner);
	    }
	    else
	    {
	        //can't call this for players because it sets viewtarget on server side screwing up the pawn rotation
            P.GameHasEnded(EndGameFocus, isWinner);	    
        }
	}   
}


function bool CheckEndGame(PlayerReplicationInfo Winner, string Reason)
{
    local Controller P;
    local bool bLastMan;

    if ( bOverTime )
    {
        if ( Numbots + NumPlayers == 0 )
            return true;
        bLastMan = true;
        foreach WorldInfo.AllControllers(class'Controller', P)
        {
            if ( (P.PlayerReplicationInfo != None) && !P.PlayerReplicationInfo.bOutOfLives )
            {
                bLastMan = false;
                break;
            }
        }
        if ( bLastMan )
            return true;
    }

    bLastMan = ( Reason ~= "LastMan" );

    if ( CheckModifiedEndGame(Winner, Reason) )
        return false;

    if ( bLastMan )
    {
        GameReplicationInfo.Winner = Winner.Team;
    }
    else
    {
        if ( Teams[1].Score == Teams[0].Score )
        {
            // Don't allow overtime with automated perf testing.
            if( bAutomatedPerfTesting )
            {
                GameReplicationInfo.Winner = Teams[0];
            }
            else
            {
                if ( !bOverTimeBroadcast )
                {
                    StartupStage = 7;
                    PlayStartupMessage();
                    bOverTimeBroadcast = true;
                }
                return false;
            }
        }
        else if ( Teams[1].Score > Teams[0].Score )
        {
            GameReplicationInfo.Winner = Teams[1];
        }
        else
        {
            GameReplicationInfo.Winner = Teams[0];
        }
    }

    EndTime = WorldInfo.RealTimeSeconds + EndTimeDelay;
    
    SetEndGameFocus(Winner);
       
    return true;
}

//called when goal score is made
function ScoreBall(UTBRBall ball, bool isBallThrowScore)
{
    local int i, GoalsScored;
    local float ppp,numtouch,maxpoints,maxper, oldTeamScore, otherTeamScore;
    local UTPlayerReplicationInfo ScorerPRI;
    local UTPlayerController PC;

	if (WorldInfo.Game.bGameEnded )
	{
	    return;
	}
	
    ScorerPRI = ball.LastHolderPRI;
   
    oldTeamScore = ScorerPRI.Team.Score;
    otherTeamScore = Teams[1 - ScorerPRI.GetTeamNum()].Score;
    
    //set team and scorer scores
    if (isBallThrowScore)
    {
        ScorerPRI.Score += 2;
        ScorerPRI.Team.Score += 3.0;        
        
        //max points to be distributed to all players who carried ball
        MaxPoints=10;
        
        //max points per player who carried ball
        MaxPer=2;       
    }
    else
    {
        ScorerPRI.Score += 5;
        ScorerPRI.Team.Score += 7.0;
        
        MaxPoints=20;
        MaxPer=5;       
    }
    
    //br logic is that when team overtakes other team score it will say 'takes the lead',
    //and on the very next score it says 'increases lead' if the scoring team is the same team which took the lead.
    if (ScorerPRI.Team.Score > otherTeamScore)
    {
        if (oldTeamScore <= otherTeamScore)
        {
            if (otherTeamScore > 0)
            {
                takesLead = true;
                increasesLead = false;
            }
            else
            {
                takesLead = false;
                increasesLead = false;            
            }
        }
        else if (takesLead)
        {
            takesLead = false;
            increasesLead = true;
        }
        else
        {
            takesLead = false;        
            increasesLead = false;        
        }
    }
    else
    {
        takesLead = false;        
        increasesLead = false;      
    }
        

    // distribute assist points to each player who held ball
    // Each player gets MaxPoints/x but it's guarenteed to be at least 1 point but no more than MaxPer points
    numtouch=0;
    for (i=0;i<ball.Assists.length;i++)
    {
        if ( (ball.Assists[i]!=None) && 
             (ball.Assists[i].PlayerReplicationInfo != none) &&
             (ScorerPRI != none) &&
             (ball.Assists[i].PlayerReplicationInfo.Team == ScorerPRI.Team) )
            numtouch = numtouch + 1.0;
    }

    ppp = MaxPoints / numtouch;
    if (ppp<1.0)
        ppp = 1.0;

    if (ppp>MaxPer)
        ppp = MaxPer;

    for (i=0;i<ball.Assists.length;i++)
    {
        if ( (ball.Assists[i]!=None) && 
             (ball.Assists[i].PlayerReplicationInfo != none) &&
             (ScorerPRI != none) &&
             (ball.Assists[i].PlayerReplicationInfo.Team == ScorerPRI.Team) )
        {
            ball.Assists[i].PlayerReplicationInfo.Score += int(ppp);
        }
    }


    GoalsScored = ScorerPRI.IncrementEventStat('EVENT_SCOREDFLAG');

    if ( GoalsScored == 3 )
    {
        ScorerPRI.IncrementEventStat('EVENT_HATTRICK');
        i = 6;
    }
    else
    {
        i = 7;
    }
    
    foreach WorldInfo.AllControllers(class'UTPlayerController', PC)
    {
        PC.ReceiveLocalizedMessage(TeamScoreMessageClass, i, ScorerPRI);        
    }
    
    ScorerPRI.bForceNetUpdate = TRUE;  
    ScorerPRI.Team.bForceNetUpdate = TRUE;

    CheckScore(ScorerPRI);
   
    if ( bOverTime )
    {
        EndGame(ScorerPRI,"timelimit");
    }
    else if ( UTGameReplicationInfo(GameReplicationInfo).bStoryMode )
    {
        if ( ScorerPRI.Team.TeamIndex == 0 )
        {
            // player team scored - increase bot skill
            AdjustedDifficulty = FMin(7.0,AdjustedDifficulty + 0.5);
        }
        else
        {
             // bot team scored - decrease bot skill
            AdjustedDifficulty = FMax(0, AdjustedDifficulty - 0.5);
        }
        AdjustedDifficulty = FClamp(AdjustedDifficulty, GameDifficulty - 1.25, GameDifficulty + 1.25);
    }
    
    AnnounceScore(ScorerPRI.Team.TeamIndex);    
}

state MatchInProgress
{
 	function Timer()
	{   
        Super.Timer();
        CheckVoteTime();
    }
}

State MatchOver
{

    function AnnounceScore(int ScoringTeam)
    {
    }

 	function Timer()
	{
        Super.Timer();
        CheckVoteTime();    
	}
}

function CheckVoteTime()
{
    local int VoteCount, i;
    
    if ((VoteCollector != none) && (NumPlayers == 1))
    { 	    
        VoteCount = 0;
        
	      for (i=0; i<VoteCollector.MapVotes.Length; ++i)
	      {
		        VoteCount += VoteCollector.MapVotes[i].NoVotes;
        }
		
        if ( 
            (VoteCount > 0) &&
            (UTGameReplicationInfo(GameReplicationInfo).MapVoteTimeRemaining > 8)
           )
        {
            UTGameReplicationInfo(GameReplicationInfo).MapVoteTimeRemaining = 8;	    
        }
    }  
}


//View next active player in PRIArray.
function bool ViewPlayer(PlayerController pc, UTBRPlayerManager playerManager)
{
    for ( playerManager.SpectatorPriIndex = playerManager.SpectatorPriIndex + 1; 
          playerManager.SpectatorPriIndex < WorldInfo.GRI.PRIArray.Length; 
          playerManager.SpectatorPriIndex++ )
    {
        if ((WorldInfo.GRI.PRIArray[playerManager.SpectatorPriIndex] == none) || 
            (Controller(WorldInfo.GRI.PRIArray[playerManager.SpectatorPriIndex].Owner) == none))
        {
            continue;
        }
              
                   
        if (CanSpectate(pc, WorldInfo.GRI.PRIArray[playerManager.SpectatorPriIndex]))
        {
            pc.SetViewTarget(WorldInfo.GRI.PRIArray[playerManager.SpectatorPriIndex]);
            playerManager.SpectatorViewTarget = pc.ViewTarget;
            playerManager.SpectatorViewTargetPRI = WorldInfo.GRI.PRIArray[playerManager.SpectatorPriIndex];
            return true;
        }
    }
    
    if (playerManager.SpectatorPriIndex >= WorldInfo.GRI.PRIArray.Length)
    {
        playerManager.SpectatorPriIndex = -1;    
    }

    return false;
}

/*
this is called by ut when spectator clicks mouse to view next objective.

br uses it to scroll though objectives, free camera, and players

default ut behavior is annoying. users are used to clicking to view players, but
now they gotta use mouse scroll wheel to view players.
most users can't even figure that out, even I had to look in the code to understand this.
*/
function ViewObjective(PlayerController PC)
{   
    local UTBRPlayerManager playerManager;
    
    playerManager = GetPlayerManager(PC.PlayerReplicationInfo);
    if (playerManager == none)
    {
       return;
    }
    
	PC.bCollideWorld = false;
    PC.SetCollision(false, false);
    playerManager.SpectatorFreeCamera = false;
    playerManager.SpectatorViewTargetPRI = none;    
    
    if ((UTBRBall(playerManager.SpectatorViewTarget) != none) || 
        (Pawn(playerManager.SpectatorViewTarget) != none) ||
        (PlayerReplicationInfo(playerManager.SpectatorViewTarget) != none))
    {
        if (ViewPlayer(PC, playerManager))
        {
            return;
        }
 
        PC.SetViewTarget(PC);
        playerManager.SpectatorFreeCamera = true;
        playerManager.SpectatorViewTarget = none;           
                   
        return;
    }
       
    if ((playerManager.SpectatorViewTarget == none) || (playerManager.SpectatorViewTarget == PC)) 
    {
        playerManager.SpectatorViewTarget = Goals[0];
        PC.SetViewTarget(Goals[0]);
    } else if (playerManager.SpectatorViewTarget == Goals[0])
    {
        PC.SetViewTarget(Goals[1]);
        playerManager.SpectatorViewTarget = Goals[1];
    } else if (playerManager.SpectatorViewTarget == Goals[1])
    {
        PC.SetViewTarget(BallHomeBase);
        playerManager.SpectatorViewTarget = BallHomeBase;
    } else
    {
        PC.SetViewTarget(TheBall);
        playerManager.SpectatorViewTarget = TheBall;     
    }
}

function string TeamColorText(int TeamNum)
{
    if (TeamNum==1)
    {
        return "Blue";
    }
    else
    {
        return "Red";
    }
}

function string DecodeEvent(name EventType, int TeamNo, string InstigatorName, string AdditionalName, class<object> AdditionalObj)
{
    if (EventType == 'flag_returned')
    {
        return InstigatorName@"returned his team's flag ("$TeamColorText(TeamNo)$")";
    }
    else if (EventType == 'flag_returned_friendly')
    {
        return InstigatorName@"made a quick return of team's flag ("$TeamColorText(TeamNo)$")";
    }
    else if (EventType == 'flag_returned')
    {
        return InstigatorName@"saved the day by returning his team's flag deep in the enemy zone ("$TeamColorText(TeamNo)$")";
    }
    else if (EventType == 'flag_captured')
    {
        return InstigatorName@"captured the enemy flag scoring a point for the"@TeamColorText(TeamNo)@"team!";
    }
    else if (EventType == 'flag_taken')
    {
        return InstigatorName@"grabbed the "$TeamColorText(TeamNo)$"flag!";
    }
    else
    {
        return super(UTTeamGame).DecodeEvent(EventType, TeamNo, InstigatorName, AdditionalName, AdditionalObj);
    }
}

function SetPlayerObjectives(bool makePathArrows, optional UTPlayerController makePathArrowsFor)
{
    local UTPlayerController PC;
    local bool makeArrows;
    
    foreach WorldInfo.AllControllers(class'UTPlayerController', PC)
    {   
        makeArrows = makePathArrows || (PC == makePathArrowsFor);
        
        if (! makeArrows)
        {
            //this is done because playercontroller will spawn path arrows every time LastAutoObjective is different from.
            //current objective, so setting LastAutoObjective to current objective here turns that off
            //we spawn arrows at start of round to help players find ball base.
            PC.LastAutoObjective = GetAutoObjectiveFor(PC);
            PC.ClientSetAutoObjective(PC.LastAutoObjective);
        }
        
        PC.CheckAutoObjective(! makeArrows);
    }
    
    RetaskBots();
}

function RetaskBots()
{
    local UTBot B;
    
    foreach WorldInfo.AllControllers(class'UTBot', B)
    {
        if (B.Squad != none)
        {
            B.Squad.Retask(B);
        }
    }
}

function Actor GetAutoObjectiveFor(UTPlayerController PC)
{
    local Actor ret;

    if (UTTeamInfo(PC.PlayerReplicationInfo.Team) != None)
    {
        if (TheBall.IsHeldByTeamOf(PC.Pawn))
        {
            
            ret = Teams[1 - PC.PlayerReplicationInfo.Team.TeamIndex].HomeBase;          
        }
        else
        {
            ret = TheBall;
        }
    }

    return ret;
    
}

simulated function Controller GetController(Pawn P)
{
    local Controller C;

    if (P == none)
    {
        return none;
    }
            
    C = P.Controller;
    
    if (P.DrivenVehicle != none)
    {
        C = P.DrivenVehicle.controller;
    }   
    
    return C;
}

simulated function UTPlayerReplicationInfo GetPlayerReplicationInfo(Pawn P)
{
    local Controller C;
    
    C = GetController(P);
       
    if (C == none)
    {
        return none;
    }
    
    return UTPlayerReplicationInfo(C.PlayerReplicationInfo);
}

//return ball actor is holding, or actor if actor is ball
function UTBRBall GetBall(Actor actor)
{
    local Pawn P;
    local UTPlayerReplicationInfo pri;
    
    if (UTBRBall(actor) != none)
    {
        return UTBRBall(actor);
    }
       
    if (Controller(actor) != none)
    {
         pri = UTPlayerReplicationInfo(Controller(actor).PlayerReplicationInfo);     
    }
    else
    {    
        P = Pawn(actor);
        
        if (P == none)
        {
            return none;
        }
        
        pri = GetPlayerReplicationInfo(P);
    }
         
    if ( pri == none)
    {
        return none;
    }
    
    return UTBRBall(pri.GetFlag());
}

// Returns whether a mutator should be allowed with this gametype
static function bool AllowMutator( string MutatorClassName )
{
    return true;
}

static function string GetVersionMessage()
{
   return "Bombing Run version " $ default.Version $ " Author Mark Caldwell Aka W@rHe@d of The Reliquary http://www.relicsut.com.";
}

//return distance bot is blocked in dir, up to max of dist.
//specify startLoc to start trace from there.
//returns 0 if no blockage found.
function float BlockedDist(Actor traceFrom, Vector dir, float dist, optional Vector startLoc, optional vector extent, optional int options)
{
    local vector hitLocation, hitNormal;
    local Actor actor;
    local bool blocked;
    local float ret;
    local Actor excludedActor;
   
    //trace() doesn't work well. seen it report a blockage when bot is just in front of goal.
    //traceActors much better as we have precise control over what's reported as a blockage.

    if (UTBRPlayerManager(traceFrom) != none)
    {    
        traceFrom = UTBRPlayerManager(traceFrom).Controller.Pawn;
    }
    
    if (Pawn(traceFrom) != none)
    {
        excludedActor = Pawn(traceFrom).Controller.MoveTarget;    
    }

    if (startLoc == vect(0,0,0))
    {
        startLoc = traceFrom.Location;
    } 
 
    foreach traceFrom.TraceActors( class'Actor', actor, hitLocation, hitNormal,
               startLoc + (dir * dist), startLoc, extent,,)
    {       
        if ((ForcedDirVolume(actor) != none) || (UTBRBall(actor) != none) || 
            (excludedActor == actor) || (UTPickupFactory(actor) != none))
        {
           //ForcedDirVolume has bBlockActors=true but doesnt' actually block
            
           continue;
        }               
        
        if (actor.bBlockActors)
        {
            blocked = true;
        }
                     
        if (blocked)
        {
            BlockingActor = actor;
            BlockingActorHitLocation = hitLocation;
            break; 
        }
    }

    if (blocked)
    {
        ret = DistanceBetweenPoints(startLoc, hitLocation);
        if (ret == 0)
        {
            ret = 1;
        }
        return ret;
    }
    
    return 0;
}

//for debugging. show path bot is taking.
function SendPlayerDebugMessage(Controller C, string msg)
{
    if (Test_ShowPath)
    {
        SendDebugMessage(C.PlayerReplicationInfo.GetPlayerAlias() $ " " $ msg);    
    }
}

//testing only. setup test settings.
exec function Test()
{
    local PlayerController PC;
       
    if ((WorldInfo.Netmode!=NM_Standalone))
    {
        return;
    }
    
    foreach WorldInfo.AllControllers(class'PlayerController', PC)
    {
        break;
    }
    
    BotsLoaded();
    AllRed();
    KillBots();    
    PC.CheatManager.God();
    PC.CheatManager.Ghost();
    GiveBall();
    //LowGrav();
    //QuadJump();
    //DestroyVehicles();
    NoNodes();
    AddBots(1);
    GoalScore = 200;
    //GoalScore = 1;
    NoWeaponVolumes();
    NoBallReset();
    VehiclesStay();
    
    ShowPath();
    NoScore();
}

//testing only. setup test settings.
exec function Test2()
{
    local PlayerController PC;
       
    if ((WorldInfo.Netmode!=NM_Standalone))
    {
        return;
    }
    
    foreach WorldInfo.AllControllers(class'PlayerController', PC)
    {
        break;
    }
                    
    
    //KillBots();
    //AddBots(1);
            
    AllRed();
    PC.CheatManager.God();

    PC.CheatManager.Ghost();
    
    GiveBall();

    GoalScore = 200;
    NoWeaponVolumes();
    NoBallReset();
    DestroyVehicles();
    LowGrav();
    
    ShowPath();
    NoScore();       
}

//testing only. setup test settings.
exec function Test3()
{
    local PlayerController PC;
       
    if ((WorldInfo.Netmode!=NM_Standalone))
    {
        return;
    }
    
    foreach WorldInfo.AllControllers(class'PlayerController', PC)
    {
        break;
    }
                    
    
    //KillBots();
    //AddBots(1);    
    PC.CheatManager.God();

    //PC.CheatManager.Ghost();
    AllRed();
    HoverBoards();
    
    GiveBall();

    GoalScore = 200;
    NoWeaponVolumes();
    NoBallReset();
    
    ShowPath();
    NoScore();
    DestroyVehicles();
    //NavigateVehicleWithBall();       
}

//testing only. toggle destroy all vehicles.
exec function DestroyVehicles()
{  
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
   
    Test_DestroyVehicles = ! Test_DestroyVehicles;
    
    SendDebugMessage("DestroyVehicles toggled " $ Test_DestroyVehicles);        
}

//testing only. toggle whether vehicles can carry ball or not.
exec function VehiclesCarryBall()
{
    local UTVehicle v;
    
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    if (bool(Settings.CarryBallInVehicles))
    {
        Settings.CarryBallInVehicles = "false";    
    }    
    else
    {
        Settings.CarryBallInVehicles = "true";     
    }
    
    ForEach WorldInfo.AllActors(class'UTVehicle', v)
    {
        if (UTVehicle_Hoverboard(v) == none)
        {        
            v.bCanCarryFlag = bool(Settings.CarryBallInVehicles);
        }
    }
    
    SendDebugMessage("VehiclesCarryBall toggled " $ Settings.CarryBallInVehicles);        
}

//toggle whether we have a blue team or not. testing only.
exec function AllRed()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    bForceAllRed = ! bForceAllRed;

    KillBots(); 
    AddBots(1);
   
    SendDebugMessage("Force all red toggled");
}

//toggle whether bots can shoot goal, forward pass, etc, or not. testing only.
exec function NoShoot()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_NoShoot = ! Test_NoShoot;
    SendDebugMessage("NoShoot toggled " $ Test_NoShoot);
}

//toggle whether bots can do special jumps or not. testing only.
exec function NoJump()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_NoJump = ! Test_NoJump;
    SendDebugMessage("NoJump toggled " $ Test_NoJump);
}

//toggle the display of bot pathing. testing only.
exec function ShowPath()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_ShowPath = ! Test_ShowPath;
    SendDebugMessage("ShowPath toggled");
}

//set low gravity. testing only.
exec function LowGrav()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    WorldInfo.WorldGravityZ = -100;    
    SendDebugMessage("Low gravity");    
}

//set quad jump. testing only.
exec function QuadJump()
{
    local UTPawn P;
    
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Settings.MaxMultiJump = "3";
    Settings.MultiJumpBoost = "50";
    
    ForEach WorldInfo.AllActors(class'UTPawn', P)
    {
        AssignJumps(P);
    } 
        
    SendDebugMessage("Quad jump");     
}

//regular gravity. testing only.
exec function RegGrav()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    WorldInfo.WorldGravityZ = -420.0;
    SendDebugMessage("Regular gravity");    
}

//set gravity. testing only.
exec function SetGrav(float newGrav)
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    WorldInfo.WorldGravityZ = newGrav;
    SendDebugMessage("Set gravity to " $ newGrav);    
}

//set multi jumps. testing only.
exec function SetMultiJump(int maxMultiJump)
{
    local UTPawn P;
    
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Settings.MaxMultiJump = string(maxMultiJump);
    
    ForEach WorldInfo.AllActors(class'UTPawn', P)
    {
        P.MaxMultiJump = maxMultiJump;
        P.MultiJumpRemaining = int(Settings.MaxMultiJump);
    }      
        
    SendDebugMessage("Set multi jump");    
}

//set jump boost. testing only.
exec function SetMultiJumpBoost(float jumpBoost)
{
    local UTPawn P;
    
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Settings.MultiJumpBoost = string(jumpBoost);
    
    ForEach WorldInfo.AllActors(class'UTPawn', P)
    {
        P.MultiJumpBoost = jumpBoost;
    }      
        
    SendDebugMessage("Set jump boost");    
}

//regular jumps. testing only.
exec function RegJump()
{
    local UTPawn P;
    
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Settings.MaxMultiJump = "1";
    Settings.MultiJumpBoost = "25";
    
    ForEach WorldInfo.AllActors(class'UTPawn', P)
    {
        P.MultiJumpBoost = int(Settings.MultiJumpBoost);
        P.MaxMultiJump = int(Settings.MaxMultiJump);
        P.MultiJumpRemaining = int(Settings.MaxMultiJump);
    }      
        
    SendDebugMessage("Regular jumps");    
}

//toggle whether hack pathing fix for high goals will be used. testing only.
exec function NoHackPath()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_NoHackPath = ! Test_NoHackPath;
    SendDebugMessage("NoHackPath toggled " $ Test_NoHackPath);
}

//testing only. toggle whether ball will be given to non bot player holding shield gun.
exec function GiveBall()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_GiveBall = ! Test_GiveBall;
    SendDebugMessage("GiveBall toggled. Use shield gun to get ball. " $ Test_GiveBall);
}

//testing only. toggle if scores can be made.
exec function NoScore()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_NoScore = ! Test_NoScore;
    SendDebugMessage("NoScore toggled " $ Test_NoScore);
}

//testing only. toggle if should show path ratings for defensive positions. lot's of messages will result.
exec function ShowPathRatings()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_ShowPathRatings = ! Test_ShowPathRatings;
    SendDebugMessage("ShowPathRatings toggled " $ Test_ShowPathRatings);
}

//testing only. toggle if ball resets when idle.
exec function NoBallReset()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_NoBallReset = ! Test_NoBallReset;
    SendDebugMessage("NoBallReset toggled " $ Test_NoBallReset);
}

//testing only. toggle whether weapons which deploy volumes can be used (slow volumes and etc). 
//it's annoying when bots use them during testing.
exec function NoWeaponVolumes()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_NoWeaponVolumes = ! Test_NoWeaponVolumes;
    SendDebugMessage("NoWeaponVolumes toggled " $ Test_NoWeaponVolumes);
}

//testing only. toggle fake clan tags on scoreboard
exec function ClanTags()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_ClanTags = ! Test_ClanTags;
    SendDebugMessage("Clantags toggled " $ Test_ClanTags);
}

//testing only. toggle if hoverboards are used.
exec function HoverBoards()
{
    local UTPawn P;
    
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    bAllowHoverboard = ! bAllowHoverboard;
    ForEach WorldInfo.AllActors(class'UTPawn', P)
    {
        AssignHoverboard(P);
    }    

    SendDebugMessage("Hoverboards toggled " $ bAllowHoverboard);
}

//toggle if bot will take EnemyGoalJumpPad. testing only.
exec function NoGoalJumpPad()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_NoGoalJumpPad = ! Test_NoGoalJumpPad;
    SendDebugMessage("NoGoalJumpPad toggled " $ Test_NoGoalJumpPad);
}

//toggle if bot can pickup bot. testing only.
exec function NoBallPickup()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_NoBallPickup = ! Test_NoBallPickup;
    SendDebugMessage("NoBallPickup toggled " $ Test_NoBallPickup);
}

//toggle if bots keep regenerating health. testing only.
exec function HealthyBots()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_HealthyBots = ! Test_HealthyBots;
    SendDebugMessage("HealthyBots toggled " $ Test_HealthyBots);
}

//testing only. output distance to ball
exec function DistToBall()
{
    local UTPlayerController PC;
    
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    foreach WorldInfo.AllControllers(class'UTPlayerController', PC)
    {    
        SendDebugMessage("Distance to ball is " $ vsize(PC.Pawn.Location - TheBall.Location));
        SendDebugMessage("Travel dist " $ TheBall.TravelDistance() $ "," $ WorldInfo.WorldGravityZ);        
    }
}

//set max score. testing only.
exec function SetScoreLimit(int scoreLimit)
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    GoalScore = scoreLimit;
    
    SendDebugMessage("Score limit set to " $ scoreLimit);
}

//when bot gets ball it will immediately use special navigation. testing only.
exec function NavigateWithBall()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    Test_NavigateWithBall = ! Test_NavigateWithBall;    
    
    SendDebugMessage("NavigateWithBall toggled " $ Test_NavigateWithBall);
}

//toggle allow exit on BotVehicle command. testing only.
exec function AllowExit()
{   
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    Test_AllowExit = ! Test_AllowExit;
    SendDebugMessage("AllowExit toggled " $ Test_AllowExit);    
}

//toggle to make bots jump a lot. testing only.
exec function JumpyBots()
{   
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    Test_JumpyBots = ! Test_JumpyBots;
    SendDebugMessage("JumpyBots toggled " $ Test_JumpyBots);    
}

//toggle if bots will always have a weapon will all ammo. testing only.
exec function BotsLoaded()
{   
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    Test_BotsLoaded = ! Test_BotsLoaded;
    SendDebugMessage("BotsLoaded toggled " $ Test_BotsLoaded);    
}

//toggle whether bots attend to power nodes or not. testing only.
exec function NoNodes()
{   
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    Test_NoNodes = ! Test_NoNodes;
    SendDebugMessage("NoNodes toggled " $ Test_NoNodes);    
}

//toggle whether bot path loop checks will be done or not. testing only.
exec function NoLoopChecks()
{   
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    Test_NoLoopChecks = ! Test_NoLoopChecks;
    SendDebugMessage("NoLoopChecks toggled " $ Test_NoLoopChecks);    
}

//toggle if bot will always find fresh path when retasking. testing only.
exec function AlwaysRepath()
{   
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    Test_AlwaysRepath = ! Test_AlwaysRepath;
    SendDebugMessage("AlwaysRepath toggled " $ Test_AlwaysRepath);    
}

//send ball home. testing only.
exec function ResetBall()
{   
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    TheBall.SendHome(none);
}

//toggle whether we will override ut's SetupSpecialPathAbilities() or not. testing only.
exec function NoSetupSpecialPathAbilities()
{   
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    Test_NoSetupSpecialPathAbilities = ! Test_NoSetupSpecialPathAbilities;
    SendDebugMessage("NoSetupSpecialPathAbilities toggled " $ Test_NoSetupSpecialPathAbilities);    
}

//position bot at ball. testing only.
exec function PutBotAtBall()
{
    local UTBot B;
       
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    foreach WorldInfo.AllControllers(class'UTBot', B)
    {
        B.Pawn.SetLocation(TheBall.Location);
        break;
    }    
}

//All current vehicles will stay and not be destroyed after timeout. testing only.
exec function VehiclesStay()
{   
    local UTVehicle v;
    
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    foreach WorldInfo.AllActors(class'UTVehicle', v)
    {
        v.ResetTime = WorldInfo.TimeSeconds + 999999;
    }
    
    SendDebugMessage("Vehicles set to stay");        
}

//testing only. set bot steering durations.
exec function SetBotSteer(float _steerOnDuration, float _steerOffDuration)
{   
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    SteerOnDuration = _steerOnDuration;
    SteerOffDuration = _steerOffDuration;

    SendDebugMessage("Set Bot Steer");    
}

//kill all power nodes. testing only
exec function KillPowerNodes()
{   
    local UTOnslaughtNodeObjective node;
    
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
  	foreach AllActors(class'UTOnslaughtNodeObjective', node)
    {
	      node.DisableObjective(none);
    }
}
	
//make bot drive vehicle. testing only.
function BotVehicle(class<UTVehicle> vc)
{
    local UTBot B;
    local UTVehicle V;
    local vector X,Y,Z;
    
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    
    foreach WorldInfo.AllControllers(class'UTBot', B)
    {
        if (UTVehicle(B.Pawn) != none)
        {
            B.LeaveVehicle(true);
        }
        
        GetAxes(B.Pawn.Rotation,X,Y,Z);    
        V = B.Pawn.Spawn(vc,,,B.Pawn.Location + (X * 800) );
        V.bTeamLocked = false;
        V.bAllowedExit = Test_AllowExit;
        Test_BotVehicle = true;
        
		V.TryToDrive(B.Pawn);
		TestBotVehicle = V;
		
		break;
    }
}

//make bot drive vehicle. testing only.
exec function BotVHov()
{
    local UTBot B;
    
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    foreach WorldInfo.AllControllers(class'UTBot', B)
    {    
        B.PerformCustomAction(B.Squad.GetOnHoverboard);
        UTVehicle(B.Pawn).bAllowedExit = false;
        Test_BotVehicle = true;
        break;
    }
}

//make bot drive vehicle. testing only.
exec function BotVTank()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    BotVehicle(class'UTVehicle_Goliath_Content');
}

//make bot drive vehicle. testing only.
exec function BotVManta()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    BotVehicle(class'UTVehicle_Manta_Content');
}

//make bot drive vehicle. testing only.
exec function BotVScor()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    BotVehicle(class'UTVehicle_Scorpion_Content');
}

//make bot drive vehicle. testing only.
exec function BotVHell()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    BotVehicle(class'UTVehicle_HellBender_Content');
}

//make bot drive vehicle. testing only.
exec function BotVRaptor()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    BotVehicle(class'UTVehicle_Raptor_Content');
}

//make bot drive vehicle. testing only.
exec function BotVViper()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    BotVehicle(class'UTVehicle_Viper_Content');
}

//make bot drive vehicle. testing only.
exec function BotVScav()
{
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    BotVehicle(class'UTVehicle_Scavenger_Content');
}

//bots will focus on your current position. testing only.
//real useful for testing bot's lobbing at certain places.
exec function SetBotFocus()
{
    local PlayerController PC;   
           
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    if (TestBotFocus != vect(0,0,0))
    {
        TestBotFocus = vect(0,0,0);
        SendDebugMessage("TestBotFocus turned off");
                       
        return;        
    }
    
    foreach WorldInfo.AllControllers(class'PlayerController', PC)
    { 
        TestBotFocus = PC.Pawn.Location;
        SendDebugMessage("TestBotFocus set to your current position");
                       
        return;
    }  
}

//display skill of all bots
exec function ShowBotSkill()
{
    local UTBot B;   
                
    foreach WorldInfo.AllControllers(class'UTBot', B)
    { 
        SendDebugMessage(B.PlayerreplicationInfo.GetPlayerAlias() $ " Skill " $ B.Skill);
    }  
}

//display bot ball lob info for your current rotation. testing only.
exec function CheckLob()
{
    local UTBRSquadAI squad;
    local PlayerController PC;
    local UTBRPlayerManager playerManager;    
           
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    squad = spawn(class'UTBRSquadAI');
    foreach WorldInfo.AllControllers(class'PlayerController', PC)
    {      
        playerManager = GetPlayerManager(PC.PlayerReplicationInfo);
        squad.TheBall = TheBall;
        playerManager.TheBall = TheBall;
        playerManager.GetLobDirection();
        break;  
    }    
}

//start/stop timer to spawn a flak shell at suggested strafe location for player blockage. testing only.
exec function SuggestStrafe()
{       
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }

    SendDebugMessage("SuggestStrafe toggled");
        
    if (IsTimerActive('SuggestStrafeTimerTest'))
    {
        ClearTimer('SuggestStrafeTimerTest');
    }
    else
    {
        SetTimer(1.0, true, 'SuggestStrafeTimerTest');    
    }
}

//used by SuggestStrafe
function SuggestStrafeTimerTest()
{
    local PlayerController PC;
    local Vector target;
    local UTBRPlayerManager playerManager;
    local UTBRSquadAI squad; 
   
    foreach WorldInfo.AllControllers(class'PlayerController', PC)
    {      
        playerManager = GetPlayerManager(PC.PlayerReplicationInfo);    
        //PC.Destination = Goals[1].Location;
        playerManager.Destination = TheBall.Location;
	    squad = spawn(class'UTBRSquadAI');
        playerManager.TurnOffBlocking();                
        target = playerManager.SuggestStrafeDestination();
       
        squad.Destroy();      
        PC.Pawn.Spawn(class'UTPRoj_FlakShardMain',,,target);
        
        //helps to see if Z is very different
        target.Z = PC.Pawn.Location.Z;
        PC.Pawn.Spawn(class'UTPRoj_FlakShardMain',,,target);
              
        break;
    }  
}

//test pathing from player current location. testing only.
//which: ALL, OBJECTIVES, or actor name
exec function TestPaths(string which)
{       
    local NavigationPoint N;
    local PlayerController pc;    
    local bool ok;
    local Actor target;
    local float dist;
    
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }
    
    which = caps(which);
    
    foreach WorldInfo.AllControllers(class'PlayerController', pc)
    {
        break;
    }
    
    if (pc == none)
    {
        SendDebugMessage("PlayerController not found");
        return;
    }     
    
    ForEach WorldInfo.AllActors(class'NavigationPoint', N)
    {
        ok = false;
        
        if (which == "OBJECTIVES")
        {
            if (
                (UTBRGoal(N) != none) || (UTOnslaughtNodeObjective(N) != none) ||
                (UTBRBallBase(N) != none)
               )
            {
                ok = true;
            }        
        }
               
        if (which == "ALL")
        {
            ok = true;
        }
               
        if (which == caps(string(N.name)))
        {
            ok = true;
        }
        
        target = N;
        
        if (which == "BALL")
        {
            target = TheBall;
            ok = true;
        }        
        
        if (ok)
        {
            dist = RouteDist(pc, target);
            
            if (dist < BAD_ROUTE_DIST)
            {
                SendDebugMessage("Path to " $ target $ " ok. MoveTarget=" $ MoveTarget $ " Path Distance=" $ dist);
            }
            else
            {
                SendDebugMessage("Path to " $ target $ " failed.");
            }
        }
        
        if (which == "BALL")
        {
            break;
        }
    }     
}

//return route dist to target for C. returns BAD_ROUTE_DIST if path fails.  
function float RouteDist(Controller C, Actor target)
{
    local float dist;
    local Actor mt;

    dist = BAD_ROUTE_DIST;
       
    mt = C.FindPathToward(target);
    MoveTarget = mt;
    
    if (mt == target)
    {
        //ut bug. when mt == target Bot.RouteDist doesn't get updated
        dist = DistanceBetween(C.Pawn, target);
    }
    else if (mt != none)
    {
        dist = C.RouteDist;
    }
    
    return dist;
}

//display version number message in console
simulated exec function BRVersion()
{  
    SendDebugMessage(class'UTBRGame'.static.GetVersionMessage());
}

//go into spectate mode. Only for admins or instant action.
simulated exec function Spectate()
{
    if ((LocalPlayerManager == none) || (LocalPlayerManager.PlayerReplicationInfo == none) || 
         (LocalPlayerManager.LocalPlayerController == none))
    {
        return;
    }  
      
    LocalPlayerManager.Spectate();
}

//go into settings screen. Only for admins or instant action.
//this was added because of epic bug wherein menu option doesn't show up
//when having multiple voted game types Online. added only as a temporary
//workaround until epic fixes.
simulated exec function BRSettings()
{
    local UTBRGameSettingsScene scene;
    
    if ((LocalPlayermanager != none) && (UTPlayerController(LocalPlayerManager.Controller) != none))
    {
        scene = UTBRGameSettingsScene(DynamicLoadObject("BombingRunGraphics.BRGameSettings", class'UTBRGameSettingsScene'));
        UTPlayerController(LocalPlayerManager.Controller).OpenUIScene(scene);
    }
}

//always been annoyed that you can't replay the campaign movies, so here's
//a command which lets you do it easy
exec function PlayMovie(string which)
{
	local UTMissionGRI MissionGRI;
	
    if (WorldInfo.Netmode!=NM_Standalone)
    {
        return;
    }	

	MissionGRI = spawn(class'UTMissionGRI');
        	
	Switch (which)
	{
        case "1": MissionGRI.PlayBinkMovie("UTCin-Intro"); break;  	
        case "2": MissionGRI.PlayBinkMovie("UTCin-Tutorial"); break;  
	    case "3": MissionGRI.PlayBinkMovie("UTCin-Combat_Briefing"); break;   
	    case "4": MissionGRI.PlayBinkMovie("UTCin-Axon_Treaty"); break;  
	    case "5": MissionGRI.PlayBinkMovie("UTCin-Necris_Attack"); break;  
	    case "6": MissionGRI.PlayBinkMovie("UTCin-NEG_Intercedes"); break;  
	    case "7": MissionGRI.PlayBinkMovie("UTCin-Malcolm_Betrayal"); break;  
	    case "8": MissionGRI.PlayBinkMovie("UTCin-Conclusion"); break;
		default: MissionGRI.PlayBinkMovie(which); break;
	}
	
	MissionGRI.destroy();
	
    SendDebugMessage("Playing movies tends to disable the ESC key, so you may have to use the 'exit' console command to exit the game.");	
}

//set debuging mode. Only for admins unless ini is setup to allow anyone.
simulated exec function SetDebug(string debugStr)
{
    if ((LocalPlayerManager == none) || (LocalPlayerManager.PlayerReplicationInfo == none) || 
         (LocalPlayerManager.LocalPlayerController == none))
    {
        return;
    }  
      
    LocalPlayerManager.SetDebug(debugStr);
}

static function string GetEndOfMatchRules(int InGoalScore, int InTimeLimit)
{
   local string ret;
   
   ret = super.GetEndOfMatchRules(InGoalScore, InTimeLimit);
 
   return ret $ default.SkillDescription;
}

function SetSkillDescription()
{
    local string desc;
    
    if (GameDifficulty == 1) desc = "Novice";
    if (GameDifficulty == 2) desc = "Average";
    if (GameDifficulty == 3) desc = "Experienced";        
    if (GameDifficulty == 4) desc = "Skilled";
    if (GameDifficulty == 5) desc = "Adept";
    if (GameDifficulty == 6) desc = "Masterful";
    if (GameDifficulty == 7) desc = "Inhuman";
    if (GameDifficulty == 8) desc = "Godlike";

    if (desc != "")
    {
        desc = " - " $ desc $ " Bots";
    }
    
    //don't display if no bots. also found that GameDifficulty is wrong sometimes if there are no bots.
    if (DesiredPlayerCount <= 1)
    {
        desc = "";
    }
    
    SkillDescription = desc;      
    default.SkillDescription = desc;   
}

//from UTTeamGame
function AnnounceScore(int ScoringTeam)
{
	local UTPlayerController PC;
	local int MessageIndex;

	if ( TeamScoreMessageClass == None )
	{
		return;
	}

	if (takesLead)
	{
		MessageIndex = 4 + ScoringTeam;
	}
	else if (increasesLead)
	{
		MessageIndex = 2 + ScoringTeam;
	}
	else
	{
		MessageIndex = ScoringTeam;
	}

	foreach WorldInfo.AllControllers(class'UTPlayerController', PC)
	{
		PC.ReceiveLocalizedMessage(TeamScoreMessageClass, MessageIndex);
	}
}

replication
{ 
    if (Role == ROLE_Authority)
        BallHomeBase, GameEnded, DebugMatchStartup, IsPendingMatch, DebugHud, SkillDescription, AdminRightsNotNeeded;
}

defaultproperties
{
   RemoteRole = ROLE_SimulatedProxy;
   bAlwaysRelevant = true;
   Version = "3.2"
   AnnouncerMessageClass=Class'UTGameContent.UTOnslaughtOrbMessage'
   TeamScoreMessageClass=Class'UTGameContent.UTTeamScoreMessage'
   HUDType=Class'BombingRun.UTBRHUD'
   Name="Default__UTBRGame"
   ObjectArchetype=UTTeamGame'UTGame.Default__UTTeamGame' 
   MapPrefixes(0)="BR"
   GameName="Bombing Run"
   Description = "Both teams seek possesion of a single Orb. Orb Carrier must take the orb to the enemy goal to score. Author: Mark Caldwell Aka W@rHe@d of The Reliquary http://www.relicsut.com."; 
   bAllowTranslocator=False
   bAllowHoverboard=True
   bMidGameHasMap=True
   Acronym="BR"
   OnlineGameSettingsClass=Class'BombingRun.UTGameSettingsBR'
   bScoreTeamKills=False
   bSpawnInTeamArea=True
   bScoreVictimsTarget=True
   TeamAIType(0)=Class'BombingRun.UTBRTeamAI'
   TeamAIType(1)=Class'BombingRun.UTBRTeamAI'
   FlagKillMessageName="ORBKILL"   
   //FlagKillMessageName="FLAGKILL"
   bUndrivenVehicleDamage=True
   bScoreDeaths=False
   EndOfMatchRulesTemplateStr_Scoring="First team to score `g wins"
   EndOfMatchRulesTemplateStr_Time="Team with most points in `t mins. wins"
   
   //unfortunately the mid game menu scoreboards are burned into the mid game menu.
   //we can't access our br scoreboard from there without replacing the whole ut3 in game menu :(
   MidgameScorePanelTag="ONSPanel"
   
   GoalScore=7
   DeathMessageClass=Class'UTGame.UTTeamDeathMessage'
   TranslocatorClass=Class'UTGameContent.UTWeap_Translocator_Content'     
   OnlineStatsWriteClass=Class'BombingRun.UTLeaderboardWriteBR'
   Test_AllowExit = true;
   SkillDescription = "";
   AdminRightsNotNeeded=false;   
}
